﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.ComponentModel;
using System.Threading;
using System.Reflection;
using System.Workflow.Runtime;
using System.Workflow.Runtime.Hosting;
using System.Workflow.Activities;
 
using System.Workflow.Activities.Rules;
using System.Workflow.ComponentModel;
using System.Workflow.ComponentModel.Compiler;
using System.Workflow.ComponentModel.Serialization;
using System.CodeDom.Compiler;

using System.Xml;
using System.IO;









/*  Workflow authoring modes: Code-separation, Code-Only and No-Code 
    Compiling .xoml files to in-memory or in-disk assemblies
    XAML activation
    Workflow serialization
    */
namespace Lessons.Workflows
{
    class Program
    {
 
        static readonly string RootFolder = Path.GetFullPath(@"..\..\");
        static readonly string NoCodeXAMLActivationFileName = RootFolder + @"NoCode_XAMLActivation.xoml";



        static void Main(string[] args)
        {
            ExecuteWorkflow(typeof(CodeOnly));                  // CodeOnly mode, the normal default mode
            ExecuteWorkflow(typeof(CodeSeparationWorkflow));    // Code-separation mode                 
            ExecuteWorkflow(NoCode_Compiled());                 // NoCode mode with run-time compilation
            ExecuteWorkflow(NoCode_XAMLActivation());           // NoCode mode with XAML activation

            SerializeWorkflow();                                // workflow serialization
            CompileXAML();                                      // compiling a .xoml file to an assembly in disk

            Console.WriteLine("Hit <ENTER> to exit");
            Console.ReadLine();
        }


        /* executes a workflow passed as a Type i.e typeof(MyWorkflow) */
        static void ExecuteWorkflow(Type WorkflowType)
        {
            using (WorkflowRuntime workflowRuntime = new WorkflowRuntime())
            { 
                AutoResetEvent waitHandle = new AutoResetEvent(false);
                workflowRuntime.WorkflowCompleted += delegate(object sender, WorkflowCompletedEventArgs e) { waitHandle.Set(); };
                workflowRuntime.WorkflowTerminated += delegate(object sender, WorkflowTerminatedEventArgs e)
                {
                    Console.WriteLine(e.Exception.Message);
                    waitHandle.Set();
                };

                WorkflowInstance instance = workflowRuntime.CreateWorkflow(WorkflowType);
                instance.Start();

                waitHandle.WaitOne();
                Console.WriteLine("----------------------------------");
            }
        }

        /* executes a workflow passed as the content of an XmlReader object */
        static void ExecuteWorkflow(XmlReader Reader)
        {
 
            using (WorkflowRuntime workflowRuntime = new WorkflowRuntime())
            {       

                AutoResetEvent waitHandle = new AutoResetEvent(false);
                workflowRuntime.WorkflowCompleted += delegate(object sender, WorkflowCompletedEventArgs e) { waitHandle.Set(); };
                workflowRuntime.WorkflowTerminated += delegate(object sender, WorkflowTerminatedEventArgs e)
                {
                    Console.WriteLine(e.Exception.Message);
                    waitHandle.Set();
                };

                WorkflowInstance instance = workflowRuntime.CreateWorkflow(Reader); 
                instance.Start();

                waitHandle.WaitOne();
                Console.WriteLine("----------------------------------");

            }
        }



        /*  Demonstrates the NoCode workflow authoring mode.
            
            Uses a string representation of a workflow which. The <x:Code> element
            defines in-line source code for a method which is linked to the event of an Activity.
         
            The string representation of the workflow is first saved to a disk file with an .xoml extension.
            Then the .xoml file is compiled, using the WorkflowCompiler class, into an in-memory assembly.
            The WorkflowCompilerParameters.GenerateInMemory property controls the type of the compiler output.
         
            The compilation creates an in-memory assembly which contains the NoCode_Compiled class, thanks
            to the x:Class="NoCode_Compiled" attribute contained in the XAML.
         
            The method returns that NoCode_Compiled compiled class using the WorkflowCompilerResults object
            returned by the WorkflowCompiler.Compile() method.
                return Results.CompiledAssembly.GetType("Lessons.Workflows.NoCode_Compiled");
         
         */
        static Type NoCode_Compiled()
        {
            string DataXML =
                    @"<?xml version=""1.0"" encoding=""utf-8""?>                       " +
                    "<SequentialWorkflowActivity                                        " +
                    "xmlns=\"http://schemas.microsoft.com/winfx/2006/xaml/workflow\"    " +
                    "xmlns:x=\"http://schemas.microsoft.com/winfx/2006/xaml\"           " +
                    "x:Class=\"Lessons.Workflows.NoCode_Compiled\" >                    " +
                    "<CodeActivity ExecuteCode=\"act_ExecuteCode\" />                   " +
                    "<x:Code>                                                           " +
                    "   <![CDATA[                                                       " +
                    "   private void act_ExecuteCode(object sender, EventArgs e)        " +
                    "   {                                                               " +
                    "      Console.WriteLine(\"Hi there, from: NoCode Compiled\");      " +
                    "  }                                                                " +
                    "   ]]>                                                             " +
                    "</x:Code>                                                          " +
                    "</SequentialWorkflowActivity>                                      "
                    ;

            /* create and save a .xoml file */
            string[] Files = { @"..\..\NoCode_Compiled.xoml" };
            File.WriteAllText(Files[0], DataXML);


            /* create and setup a WorkflowCompiler */
            WorkflowCompiler compiler = new WorkflowCompiler();
            WorkflowCompilerParameters Params = new WorkflowCompilerParameters();
            Params.GenerateInMemory = true;


            /* compile the file(s) and check for any errors */
            WorkflowCompilerResults Results = compiler.Compile(Params, Files);

            if (Results.Errors.Count > 0)
                foreach (CompilerError ce in Results.Errors)
                    Console.WriteLine(ce.ErrorText);

            /* load the NoCodeCompiledWorkflow class which is contained in the dynamically compiled assembly */
            return Results.CompiledAssembly.GetType("Lessons.Workflows.NoCode_Compiled");
        }


        /*  Demonstrates the so-called XAML activation.  
         
            XAML activation involves no compiling at all. 
            The .xoml file is just loaded into an XmlReader object and then it sent for execution.         
            Since no compilation takes place the x:Class="..." attribute is illegal.
         
            This method saves a string representation of a workflow into a disk file.
            It then loads the file into a FileStream object and returns an XmlReader object.
         
            The .xoml uses a custom activity class, the TextActivity class and a DelayActivity class.
            The line
                xmlns:ns0=\"clr-namespace:Lessons.Workflows; Assembly=CodeOnly, Version=1.0.0.0, Culture=neutral, PublicKeyToken=null"
            defines the ns0 alias for the Lessons.Workflows namespace.
         
            Also the .xoml file passes some values to Activity properties used by the workflow.
         
         */
        static XmlReader NoCode_XAMLActivation()
        {

            string DataXML =
                @"<?xml version=""1.0"" encoding=""utf-8""?>                       " +
                "<SequentialWorkflowActivity                                        " + 
                "xmlns=\"http://schemas.microsoft.com/winfx/2006/xaml/workflow\"    " +
                "xmlns:x=\"http://schemas.microsoft.com/winfx/2006/xaml\"           " +
                "xmlns:ns0=\"clr-namespace:Lessons.Workflows; Assembly=AuthoringModes, Version=1.0.0.0, Culture=neutral, PublicKeyToken=null\" 	 " +
                //"x:Class=\"Lessons.Workflows.DynamicWorkflow\"     " +
                 ">                                                                  " +
                "<ns0:TextActivity Text=\"NoCode XAML Activation\" />               " +
                "<DelayActivity TimeoutDuration=\"00:00:02\"  />                    " +
                "</SequentialWorkflowActivity>                                      "
                ;
 
            /* create and save a .xoml file */
            File.WriteAllText(NoCodeXAMLActivationFileName, DataXML);

            FileStream FS = File.OpenRead(NoCodeXAMLActivationFileName);
            return XmlReader.Create(FS);
        }


        /*  Demonstrates workflow serialization.
            
            Workflow serialization is done by using the WorkflowMarkupSerializer class.
                workflowSerializer.Serialize(xmlWriter, workflowInstance.GetWorkflowDefinition());
         */
        static void SerializeWorkflow()
        {
            WorkflowMarkupSerializer WS = new WorkflowMarkupSerializer();

            using (WorkflowRuntime workflowRuntime = new WorkflowRuntime())
            {
                using (StringWriter SW = new StringWriter())
                {
                    using (XmlWriter XW = XmlWriter.Create(SW))
                    {
                        Console.WriteLine("Serializing a Workflow");
                        Console.WriteLine("==================================");
                        WorkflowInstance instance = workflowRuntime.CreateWorkflow(typeof(CodeSeparationWorkflow));
                        WS.Serialize(XW, instance.GetWorkflowDefinition());
                        Console.WriteLine(SW.ToString());
                        Console.WriteLine("----------------------------------");
                    }
                }
            }
        }




        /*  Demonstrates compiling a .xoml file to an assembly which is then saved to disk.         
         
            For the compilation to succeed the .xoml has to contain the x:Class attribute
            pointing to a full qualified (non-existed though) class name.
                x:Class= "Lessons.Workflows.DynamicWorkflow"

         */
        static void CompileXAML()
        {
            string DataXML =
                @"<?xml version=""1.0"" encoding=""utf-8""?>                       " +
                "<SequentialWorkflowActivity                                        " +
                "xmlns=\"http://schemas.microsoft.com/winfx/2006/xaml/workflow\"    " +
                "xmlns:x=\"http://schemas.microsoft.com/winfx/2006/xaml\"           " +
                "xmlns:ns0=\"clr-namespace:Lessons.Workflows; Assembly=AuthoringModes, Version=1.0.0.0, Culture=neutral, PublicKeyToken=null\" 	 " +
                "x:Class=\"Lessons.Workflows.DynamicWorkflow\"     " +
                 ">                                                                  " +
                "<ns0:TextActivity Text=\"NoCode XAML Activation\" />               " +
                "<DelayActivity TimeoutDuration=\"00:00:02\"  />                    " +
                "</SequentialWorkflowActivity>                                      "
                ;

            /* create and save a .xoml file */
            File.WriteAllText(NoCodeXAMLActivationFileName, DataXML);

            string[] Files = { NoCodeXAMLActivationFileName };

            /* create and setup a WorkflowCompiler */
            WorkflowCompiler compiler = new WorkflowCompiler();
            WorkflowCompilerParameters Params = new WorkflowCompilerParameters(new string[] { "AuthoringModes.exe" });
            Params.OutputAssembly = RootFolder + "NoCodeXAMLActivationCompiled.dll";


            WorkflowCompilerResults Results = compiler.Compile(Params, Files);

            if (Results.Errors.Count > 0)
                foreach (CompilerError ce in Results.Errors)
                    Console.WriteLine(ce.ErrorText);


            Console.WriteLine("Compiling a Workflow from a .xoml file at run-time");
            Console.WriteLine("==================================");
            Console.WriteLine(".xoml file compiled to: {0}", Params.OutputAssembly);
            Console.WriteLine("----------------------------------");
        }

    }





    /*  A workflow class to be used in demonstrating the CodeOnly authoring mode.
        The constructor performs the initialization usually done by the InitializeComponent() method 
        the IDE puts in workflow classes.
        */
    public class  CodeOnly : SequentialWorkflowActivity
    {
        public CodeOnly()
        {
            CanModifyActivities = true;

            CodeActivity a = new CodeActivity();
            a.ExecuteCode += new System.EventHandler(this.act_ExecuteCode);
            this.Activities.Add(a);

            CanModifyActivities = false;
        }

        private void act_ExecuteCode(object sender, EventArgs e)
        {
            Console.WriteLine("Hi there, from: CodeOnly"); 
        }
    }


    /* A custom activity class used by the above examples in .xoml */
    public class TextActivity : Activity
    {
        public static DependencyProperty TextProperty = DependencyProperty.Register("Text", typeof(string), typeof(TextActivity));
        
        public TextActivity()
        {
        }

        protected override ActivityExecutionStatus Execute(ActivityExecutionContext context)
        {
            Console.WriteLine("Hi there, from: {0}", Text);
            return ActivityExecutionStatus.Closed;
        }

        public string Text
        {
            get {  return ((string)(base.GetValue(TextProperty)));  }
            set {  base.SetValue(TextProperty, value);  }
        }
    }
}


 
 